<?php

namespace app\controllers;

use Yii;
use yii\web\Controller;
use yii\web\NotFoundHttpException;
use yii\filters\AccessControl;
use yii\data\ActiveDataProvider;
use app\models\Booking;
use app\models\RoomType;
use app\models\Review;

class BookingController extends Controller
{
    /**
     * {@inheritdoc}
     */
    public function behaviors()
    {
        return [
            'access' => [
                'class' => AccessControl::class,
                'rules' => [
                    [
                        'allow' => true,
                        'roles' => ['@'],
                    ],
                ],
            ],
        ];
    }

    /**
     * Список бронирований пользователя
     */
    public function actionIndex()
    {
        $query = Booking::find()
            ->where(['user_id' => Yii::$app->user->id])
            ->orderBy(['created_at' => SORT_DESC]);

        $dataProvider = new ActiveDataProvider([
            'query' => $query,
            'pagination' => [
                'pageSize' => 10,
            ],
        ]);

        return $this->render('index', [
            'dataProvider' => $dataProvider,
        ]);
    }

    /**
     * Создание нового бронирования
     */
    public function actionCreate()
    {
        $model = new Booking();
        
        // Объявляем переменные в начале метода
        $roomTypes = RoomType::find()->all();
        $minDate = date('Y-m-d');
        $maxDate = date('Y-m-d', strtotime('+1 year'));

        if ($model->load(Yii::$app->request->post())) {
            // Логирование данных для отладки
            Yii::info('Данные бронирования: ' . print_r($model->attributes, true));
            
            if ($model->save()) {
                Yii::$app->session->setFlash('success', 'Заявка на бронирование успешно создана!');
                return $this->redirect(['index']);
            } else {
                // Формируем понятное сообщение об ошибке
                $errorMessages = [];
                foreach ($model->errors as $attribute => $errors) {
                    $errorMessages[] = $model->getAttributeLabel($attribute) . ': ' . implode(', ', $errors);
                }
                
                Yii::$app->session->setFlash('error', 
                    'Произошла ошибка при создании заявки:<br>' . implode('<br>', $errorMessages));
            }
        }

        return $this->render('create', [
            'model' => $model,
            'roomTypes' => $roomTypes,
            'minDate' => $minDate,
            'maxDate' => $maxDate,
        ]);
    }

    /**
     * Просмотр бронирования
     */
    public function actionView($id)
    {
        $booking = $this->findModel($id);
        
        if ($booking->user_id !== Yii::$app->user->id) {
            throw new NotFoundHttpException('Заявка не найдена');
        }

        return $this->render('view', [
            'booking' => $booking,
        ]);
    }

    /**
     * Создание отзыва
     */
    public function actionReview($id)
    {
        $booking = $this->findModel($id);
        
        if ($booking->user_id !== Yii::$app->user->id) {
            throw new NotFoundHttpException('Заявка не найдена');
        }

        if (!$booking->canReview()) {
            Yii::$app->session->setFlash('error', 'Вы не можете оставить отзыв для этого бронирования.');
            return $this->redirect(['view', 'id' => $id]);
        }

        $model = new Review();

        if ($model->load(Yii::$app->request->post())) {
            $model->booking_id = $id;
            if ($model->save()) {
                Yii::$app->session->setFlash('success', 'Отзыв успешно добавлен!');
                return $this->redirect(['view', 'id' => $id]);
            }
        }

        return $this->render('review', [
            'booking' => $booking,
            'model' => $model,
        ]);
    }

    /**
     * Отмена бронирования
     */
    public function actionCancel($id)
    {
        $booking = $this->findModel($id);
        
        if ($booking->user_id !== Yii::$app->user->id) {
            throw new NotFoundHttpException('Заявка не найдена');
        }

        if ($booking->status === Booking::STATUS_NEW || $booking->status === Booking::STATUS_PROCESSING) {
            $booking->status = Booking::STATUS_CANCELLED;
            if ($booking->save()) {
                Yii::$app->session->setFlash('success', 'Бронирование успешно отменено.');
            } else {
                Yii::$app->session->setFlash('error', 'Произошла ошибка при отмене бронирования.');
            }
        } else {
            Yii::$app->session->setFlash('error', 'Невозможно отменить бронирование с текущим статусом.');
        }

        return $this->redirect(['index']);
    }

    /**
     * Тестовый метод для отладки
     */
    public function actionTest()
    {
        $model = new Booking();
        $model->user_id = Yii::$app->user->id;
        $model->room_type_id = 1; // ID существующего типа номера
        $model->check_in_date = date('Y-m-d');
        $model->check_out_date = date('Y-m-d', strtotime('+2 days'));
        $model->adults_count = 2;
        $model->children_count = 0;
        $model->payment_method = 'cash';
        $model->status = Booking::STATUS_NEW;
        
        echo "Проверка валидации модели:<br>";
        if ($model->validate()) {
            echo "Валидация пройдена<br>";
            if ($model->save()) {
                echo "Сохранение успешно! ID: " . $model->id;
            } else {
                echo "Ошибка сохранения: " . print_r($model->errors, true);
            }
        } else {
            echo "Ошибки валидации: " . print_r($model->errors, true);
        }
    }

    /**
     * Отладочный метод
     */
    public function actionDebug()
    {
        // Проверка подключения к базе данных
        try {
            $connection = Yii::$app->db;
            $tables = $connection->createCommand("SHOW TABLES")->queryColumn();
            echo "Подключение к БД успешно. Таблицы: " . implode(', ', $tables) . "<br><br>";
        } catch (\Exception $e) {
            echo "Ошибка подключения к БД: " . $e->getMessage() . "<br><br>";
        }
        
        // Проверка существования таблицы booking
        try {
            $columns = Yii::$app->db->createCommand("DESCRIBE {{%booking}}")->queryAll();
            echo "Структура таблицы booking:<br>";
            echo "<pre>";
            print_r($columns);
            echo "</pre>";
        } catch (\Exception $e) {
            echo "Таблица booking не существует или ошибка: " . $e->getMessage() . "<br><br>";
        }
        
        // Проверка существования таблицы room_type
        try {
            $roomTypes = RoomType::find()->all();
            echo "Типы номеров в базе: " . count($roomTypes) . "<br>";
            echo "<pre>";
            foreach ($roomTypes as $roomType) {
                echo "ID: {$roomType->id}, Название: {$roomType->name}, Макс. гостей: {$roomType->max_guests}<br>";
            }
            echo "</pre>";
        } catch (\Exception $e) {
            echo "Ошибка при получении типов номеров: " . $e->getMessage() . "<br><br>";
        }
    }

    /**
     * Поиск модели бронирования
     */
    protected function findModel($id)
    {
        $model = Booking::findOne($id);
        
        if ($model === null) {
            throw new NotFoundHttpException('Заявка не найдена');
        }
        
        return $model;
    }
}