<?php

namespace app\models;

use Yii;
use yii\db\ActiveRecord;

/**
 * Class RoomType
 * @package app\models
 * 
 * @property integer $id
 * @property string $name
 * @property string $description
 * @property float $price_per_night
 * @property integer $max_guests
 * @property string $bed_type
 * @property string $room_size
 * @property string $amenities
 * @property string $image_url
 * @property integer $created_at
 * @property integer $updated_at
 */
class RoomType extends ActiveRecord
{
    /**
     * {@inheritdoc}
     */
    public static function tableName()
    {
        return '{{%room_type}}';
    }

    /**
     * {@inheritdoc}
     */
    public function rules()
    {
        return [
            [['name', 'description', 'price_per_night', 'max_guests', 'bed_type'], 'required'],
            [['description', 'amenities'], 'string'],
            [['price_per_night'], 'number', 'min' => 0],
            [['max_guests'], 'integer', 'min' => 1],
            [['name', 'bed_type', 'room_size', 'image_url'], 'string', 'max' => 255],
            [['created_at', 'updated_at'], 'safe'],
        ];
    }

    /**
     * {@inheritdoc}
     */
    public function attributeLabels()
    {
        return [
            'id' => 'ID',
            'name' => 'Название',
            'description' => 'Описание',
            'price_per_night' => 'Цена за ночь',
            'max_guests' => 'Максимум гостей',
            'bed_type' => 'Тип кровати',
            'room_size' => 'Размер номера',
            'amenities' => 'Удобства',
            'image_url' => 'Фото',
            'created_at' => 'Дата создания',
            'updated_at' => 'Дата обновления',
        ];
    }

    /**
     * {@inheritdoc}
     */
    public function beforeSave($insert)
    {
        if (parent::beforeSave($insert)) {
            if ($insert) {
                $this->created_at = time();
            }
            $this->updated_at = time();
            return true;
        }
        return false;
    }

    /**
     * Получить форматированную цену
     * @return string
     */
    public function getFormattedPrice()
    {
        return number_format($this->price_per_night, 0, ',', ' ') . ' ₽';
    }

    /**
     * Получить удобства в виде массива
     * @return array
     */
    public function getAmenitiesArray()
    {
        return !empty($this->amenities) ? explode(', ', $this->amenities) : [];
    }
}