<?php

namespace app\models;

use Yii;
use yii\db\ActiveRecord;
use yii\web\IdentityInterface;

/**
 * User model
 *
 * @property integer $id
 * @property string $username
 * @property string $email
 * @property string $password_hash
 * @property string $full_name
 * @property string $passport_series
 * @property string $passport_number
 * @property string $address
 * @property string $phone
 * @property string $auth_key
 * @property integer $created_at
 * @property integer $updated_at
 */
class User extends ActiveRecord implements IdentityInterface
{
    const SCENARIO_REGISTER = 'register';
    const SCENARIO_LOGIN = 'login';
    const SCENARIO_UPDATE = 'update';

    public $password;
    public $password_repeat;

    /**
     * {@inheritdoc}
     */
    public static function tableName()
    {
        return '{{%user}}';
    }

    /**
     * {@inheritdoc}
     */
    public function rules()
    {
        return [
            // Регистрация
            [['username', 'email', 'full_name', 'passport_series', 'passport_number', 'address', 'phone', 'password', 'password_repeat'], 'required', 'on' => self::SCENARIO_REGISTER],
            
            // Вход
            [['email', 'password'], 'required', 'on' => self::SCENARIO_LOGIN],
            
            // Общие правила
            ['username', 'match', 'pattern' => '/^[a-zA-Z0-9]+$/', 'message' => 'Логин может содержать только латинские буквы и цифры'],
            ['username', 'string', 'min' => 3, 'max' => 50],
            ['username', 'unique', 'message' => 'Этот логин уже занят'],
            
            ['email', 'email', 'message' => 'Введите корректный email'],
            ['email', 'unique', 'message' => 'Этот email уже зарегистрирован'],
            
            ['full_name', 'match', 'pattern' => '/^[а-яА-ЯёЁ\s\-]+$/u', 'message' => 'ФИО может содержать только русские буквы, пробелы и дефисы'],
            ['full_name', 'string', 'min' => 5, 'max' => 150],
            
            ['passport_series', 'match', 'pattern' => '/^\d{4}$/', 'message' => 'Серия паспорта должна состоять из 4 цифр'],
            ['passport_number', 'match', 'pattern' => '/^\d{6}$/', 'message' => 'Номер паспорта должен состоять из 6 цифр'],
            
            ['address', 'string', 'min' => 10, 'max' => 500],
            
            ['phone', 'match', 'pattern' => '/^\+7\(\d{3}\)-\d{3}-\d{2}-\d{2}$/', 'message' => 'Телефон должен быть в формате +7(XXX)-XXX-XX-XX'],
            
            ['password', 'string', 'min' => 7, 'message' => 'Пароль должен содержать минимум 7 символов'],
            ['password_repeat', 'compare', 'compareAttribute' => 'password', 'message' => 'Пароли не совпадают', 'on' => self::SCENARIO_REGISTER],
            
            [['created_at', 'updated_at'], 'safe'],
        ];
    }

    /**
     * {@inheritdoc}
     */
    public function attributeLabels()
    {
        return [
            'id' => 'ID',
            'username' => 'Логин',
            'email' => 'Email',
            'password' => 'Пароль',
            'password_repeat' => 'Повторите пароль',
            'full_name' => 'ФИО',
            'passport_series' => 'Серия паспорта',
            'passport_number' => 'Номер паспорта',
            'address' => 'Адрес проживания',
            'phone' => 'Телефон',
            'created_at' => 'Дата регистрации',
            'updated_at' => 'Дата обновления',
        ];
    }

    /**
     * {@inheritdoc}
     */
    public function beforeSave($insert)
    {
        if (parent::beforeSave($insert)) {
            if ($insert) {
                $this->created_at = time();
                $this->auth_key = Yii::$app->security->generateRandomString();
            }
            $this->updated_at = time();
            
            if ($this->password) {
                $this->password_hash = Yii::$app->security->generatePasswordHash($this->password);
            }
            
            return true;
        }
        return false;
    }

    /**
     * {@inheritdoc}
     */
    public static function findIdentity($id)
    {
        return static::findOne($id);
    }

    /**
     * {@inheritdoc}
     */
    public static function findIdentityByAccessToken($token, $type = null)
    {
        return null;
    }

    /**
     * {@inheritdoc}
     */
    public function getId()
    {
        return $this->id;
    }

    /**
     * {@inheritdoc}
     */
    public function getAuthKey()
    {
        return $this->auth_key;
    }

    /**
     * {@inheritdoc}
     */
    public function validateAuthKey($authKey)
    {
        return $this->auth_key === $authKey;
    }

    /**
     * Finds user by username
     *
     * @param string $username
     * @return static|null
     */
    public static function findByUsername($username)
    {
        return static::findOne(['username' => $username]);
    }

    /**
     * Finds user by email
     *
     * @param string $email
     * @return static|null
     */
    public static function findByEmail($email)
    {
        return static::findOne(['email' => $email]);
    }

    /**
     * Validates password
     *
     * @param string $password password to validate
     * @return bool if password provided is valid for current user
     */
    public function validatePassword($password)
    {
        return Yii::$app->security->validatePassword($password, $this->password_hash);
    }
}